from pyparsing import Literal

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector
from codeable_detectors.evidences import FailedEvidence, Evidence, LinkEvidence
from codeable_detectors.pyparsing_patterns import ID, qualified_id, round_braces_block


class JavaConnectWithJDBC(AtLeastOneFileMatchesDetector):
    def __init__(self, driver=None):
        super().__init__()
        self.file_endings = ["java"]
        self.get_connection_pattern = (qualified_id + Literal("=") + ID + Literal(".") + Literal("getConnection") +
                                       round_braces_block + Literal(";"))
        self.driver = driver

    def detect_in_context(self, ctx, **kwargs):
        matches = []
        connection_matches = ctx.matches_pattern(self.get_connection_pattern)
        if not connection_matches:
            return FailedEvidence("no jdbc getConnection call found")
        statement_detected = False
        for connection_match in connection_matches:
            connection_var = connection_match.text[:connection_match.text.find("=")].strip()

            statement_pattern = (qualified_id + Literal("=") + Literal(connection_var) + Literal(".") +
                                 (Literal("prepareStatement") | Literal("createStatement") | Literal(
                                     "callableStatement")) +
                                 round_braces_block + Literal(";"))
            statement_matches = ctx.matches_pattern(statement_pattern)
            if statement_matches:
                statement_detected = True
                matches.append(connection_match)
                matches.extend(statement_matches)

        if not statement_detected:
            return FailedEvidence("no jdbc getConnection with matching statement call found")

        if self.driver is not None:
            driver_matches = ctx.matches_pattern(Literal(self.driver))
            if not driver_matches:
                return FailedEvidence("jdbc driver '" + self.driver + "' not found")
            matches = matches + driver_matches

        return Evidence(matches)


class JavaMySQLJDBCLink(JavaConnectWithJDBC):
    def __init__(self):
        super().__init__("com.mysql.jdbc.Driver")

    def detect_in_context(self, ctx, **kwargs):
        evidence = super().detect_in_context(ctx, **kwargs)
        if not evidence.has_succeeded():
            return evidence

        return LinkEvidence(evidence.matches).set_properties(detector_link_types=["jdbc"],
                                                             detector_technology_types=["java", "jdbc", "mysql"],
                                                             kwargs=kwargs)
